<?php

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

class Kernel extends ConsoleKernel
{
    /**
     * Define the application's command schedule.
     *
     * @param  \Illuminate\Console\Scheduling\Schedule  $schedule
     * @return void
     */
    protected function schedule(Schedule $schedule)
    {
        $env = config('app.env');
        $email = config('mail.username');

        if ($env === 'live') {

            // =========================
            // BACKUP SCHEDULE
            // =========================
            // Read admin-defined backup schedule from 'settings' table
            $backupInterval = \DB::table('settings')->where('key', 'backup_interval')->value('value') ?? 'daily';
            $backupTime = \DB::table('settings')->where('key', 'backup_time')->value('value') ?? '01:30';

            switch ($backupInterval) {
                case 'hourly':
                    $schedule->command('backup:run')->hourly()->emailOutputTo($email);
                    $schedule->command('backup:clean')->hourly()->emailOutputTo($email);
                    break;

                case 'daily':
                    $schedule->command('backup:run')->dailyAt($backupTime)->emailOutputTo($email);
                    $schedule->command('backup:clean')->dailyAt('01:00')->emailOutputTo($email); // cleaning time fixed at 1 AM
                    break;

                case 'weekly':
                    $schedule->command('backup:run')->weeklyOn(1, $backupTime)->emailOutputTo($email); // Monday
                    $schedule->command('backup:clean')->weeklyOn(1, '01:00')->emailOutputTo($email);
                    break;

                default:
                    // No backup scheduled
                    break;
            }

            // =========================
            // RECURRING TASKS
            // =========================
            $schedule->command('pos:generateSubscriptionInvoices')->dailyAt('23:30');
            $schedule->command('pos:updateRewardPoints')->dailyAt('23:45');
            $schedule->command('pos:autoSendPaymentReminder')->dailyAt('08:00');
        }

        if ($env === 'demo') {
            // IMPORTANT: Demo environment resets business data every 3 hours
            $schedule->command('pos:dummyBusiness')
                ->cron('0 */3 * * *')
                ->emailOutputTo($email);
        }
    }

    /**
     * Register the commands for the application.
     *
     * @return void
     */
    protected function commands()
    {
        $this->load(__DIR__.'/Commands');

        require base_path('routes/console.php');
    }
}
